/* Copyright (C) 2017-2018 RealVNC Ltd.  All Rights ResereturnValueed.
 */

/* This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <cstdio>
#include <string.h>
#include <gst/app/gstappsink.h>
#include <gst/app/gstappsrc.h>
#include <cmath>

#include <vnccommon/StringUtils.h>

#include "vnccommondecoderapplicationcontextimpl.h"

#include "GStreamerStream.h"

static GMainContext *contextBusLoop = NULL;
static GMainLoop* messageBusLoop = NULL;

gboolean bus_signal(GstBus *bus, GstMessage *message, gpointer user_data)
{
    (void)user_data;
    (void)bus;
    if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR)
    {
        printf("GST_MESSAGE_ERROR received");
    }
    else if (GST_MESSAGE_TYPE(message) == GST_MESSAGE_STATE_CHANGED)
    {
        printf("GST_MESSAGE_STATE_CHANGED received");
    }
    else
    {
        printf("\n\nbus_signal AT LEAST WE RECEIVED SOMETHING!\n\n");
    }
    return true;
}

void* mainLoop(void* inData)
{
    (void)inData;

    // create GMainContext
    contextBusLoop = g_main_context_new();
    if (contextBusLoop != NULL)
    {
        // create GMainLoop
        messageBusLoop = g_main_loop_new(contextBusLoop, FALSE);
        if (messageBusLoop != NULL)
        {
            g_main_context_push_thread_default(contextBusLoop);
            // this call is blocking until g_main_loop_quit called
            printf("g_main_loop_run\n");
            g_main_loop_run(messageBusLoop);
        }
        else
        {
            printf("mainLoop()  g_main_loop_new failed\n");
        }

        printf("Out of g_main_loop_run\n");
        // unref context
        g_main_context_pop_thread_default(contextBusLoop);
        g_main_context_unref(contextBusLoop);
        contextBusLoop = NULL;
    }
    else
    {
        printf("mainLoop()  g_main_context_new failed\n");
    }
    return NULL;
}

GStreamerStream::GStreamerStream(
        StreamID streamId,
        vnccommon::LogHandler::Tagged& logHandler,
        VNCCommonDecoderMediaType mediaType,
        VNCCommonDecoderApplicationContext context)

    : mId(streamId),
      mLog(logHandler.appendTag(vnccommon::StringUtils::build()
                << "Stream/"
                << streamId.toString())),
      mMediaType(mediaType),
      mContext(context),
      mGstPipeline(NULL),
      mGstSink(NULL),
      mGstVolume(NULL),
      mFramesSinceLastLog(0)
{
    mLog.info("Creating stream %s, context address is %p", streamId.toString().c_str(), mContext);

    pthread_t gstreamer_thread;
    if (0 != pthread_create(&gstreamer_thread, NULL, mainLoop, NULL))
    {
        mLog.info("Couldn't spawn thread");
    }

    memset(&mVideoMode, 0, sizeof(mVideoMode));
}

GStreamerStream::~GStreamerStream()
{
    gst_element_set_state(mGstPipeline, GST_STATE_NULL);
    gst_object_unref(mGstPipeline);
}

void GStreamerStream::setAudioFormat(const VNCCommonDecoderAudioFormatDetailsLPCM& format)
{
    // Copy over
    mAudioFormat = format;
}

void GStreamerStream::setVideoMode(const VNCCommonDecoderVideoMode& mode)
{
    mVideoMode = mode;
}

void GStreamerStream::setOverlayRectangle(const VNCRectangle& rect)
{
    mLog.info("GStreamerStream::setOverlayRectangle");
    if (mMediaType != VNCCommonDecoderMediaTypeVideo)
        return;

    if (mOverlayRectangle.topLeft.x == rect.topLeft.x ||
        mOverlayRectangle.topLeft.y == rect.topLeft.y ||
        mOverlayRectangle.bottomRight.x == rect.bottomRight.x ||
        mOverlayRectangle.bottomRight.y == rect.bottomRight.y)
    {
        return;
    }

    mOverlayRectangle = rect;

    if (!mVideoMode.width || !mVideoMode.height)
    {
        mVideoMode.width = mOverlayRectangle.bottomRight.x - mOverlayRectangle.topLeft.x;
        mVideoMode.height = mOverlayRectangle.bottomRight.y - mOverlayRectangle.topLeft.y;
    }

#ifdef BOARD_imx6
    if (mGstSink)
    {
        g_object_set(G_OBJECT(mGstSink), "window-x-coord", mOverlayRectangle.topLeft.x, NULL);
        g_object_set(G_OBJECT(mGstSink), "window-y-coord", mOverlayRectangle.topLeft.y, NULL);
        g_object_set(G_OBJECT(mGstSink), "window-width", mOverlayRectangle.bottomRight.x - mOverlayRectangle.topLeft.x, NULL);
        g_object_set(G_OBJECT(mGstSink), "window-height", mOverlayRectangle.bottomRight.y - mOverlayRectangle.topLeft.y, NULL);
    }
#endif

}

void GStreamerStream::setOverlayVisibility(bool visible)
{
    mLog.info("GStreamerStream::setOverlayVisibility");
#ifdef BOARD_imx6
    if (mMediaType == VNCCommonDecoderMediaTypeVideo)
    {
        int fb1 = open("/dev/fb1", O_RDWR);
        if (visible)
        {
            for (int i = 0; i < mResolution.y; i++)
            {
                unsigned char v[mResolution.x * 4];
                memset(&v[0], 0, 4 * mResolution.x);
                write(fb1, &v[0], 4 * mResolution.x);
            }
        }
        ioctl(fb1, FBIOBLANK, visible ? FB_BLANK_UNBLANK : FB_BLANK_NORMAL);
        close(fb1);
    }
#else
    (void) visible;
#endif
}

void GStreamerStream::streamPayload(vnc_uint8_t *data, vnc_size_t length)
{
    GstBuffer *buffer = gst_buffer_new_and_alloc(length);
/*
    FILE *foo = fopen("/tmp/foo_buf.log", "a+b");
    if (foo)
    {
        fwrite(data, length, 1, foo);
        fclose(foo);
    }
*/
    memcpy(GST_BUFFER_DATA(buffer), data, length);
    int ret = gst_app_src_push_buffer(GST_APP_SRC(mGstAppSrc), buffer);
    mLog.info("GStreamerStream::streamPayload length = %d ret = %d\n", length, ret);
}

VNCCommonDecoderError GStreamerStream::setupGStreamer()
{
    mLog.info("GStreamerStream::setupGStreamer");
    GError *error = NULL;
    gboolean returnValue = false;
    GstElement *sink = NULL;

    GstElement *decoder = NULL;
    if (mGstPipeline)
    {
        // Already setup
        return VNCCommonDecoderErrorNone;
    }

    returnValue = gst_init_check(NULL, NULL, &error);
    if (!returnValue)
    {
        mLog.info("Failed to initialise GStreamer");
        return VNCCommonDecoderErrorResourceError;
    }

    // Setup pipeline
    mGstPipeline = gst_pipeline_new("mlink_h264_pipeline");
    if (!mGstPipeline)
    {
        mLog.info("Failed to initialise pipeline");
        return VNCCommonDecoderErrorResourceError;
    }

    bus = gst_pipeline_get_bus(GST_PIPELINE(mGstPipeline));
    g_assert(bus);

    /* add watch for messages */
    gst_bus_add_signal_watch(bus);
    g_signal_connect(bus, "message",
                                      G_CALLBACK(bus_signal), NULL);

    mGstAppSrc = gst_element_factory_make("appsrc", "src");
    if (!mGstAppSrc)
    {
        mLog.info("Failed to initialise source");
        return VNCCommonDecoderErrorResourceError;
    }

    g_object_set(G_OBJECT(mGstAppSrc), "stream-type", 0 /*GST_APP_STREAM_TYPE_STREAM*/, NULL);
    g_object_set(G_OBJECT(mGstAppSrc), "max-latency", (gint64)1000000000, NULL);

    //GstCaps* caps = gst_caps_new_simple("video/x-h264", NULL);

    GstCaps* caps = gst_caps_new_simple("video/x-h264",
                                        "width", G_TYPE_INT, mVideoMode.width,
                                        "height", G_TYPE_INT, mVideoMode.height,
                                        "framerate", GST_TYPE_FRACTION, 0, 1,
                                        NULL);

    mLog.info("Height = %d Width = %d", mVideoMode.height, mVideoMode.width);

    if (caps != NULL)
    {
        gst_app_src_set_caps(GST_APP_SRC(mGstAppSrc), caps);
        gst_caps_unref(caps);
    }

    g_object_set(G_OBJECT(mGstAppSrc),
            "is-live",      TRUE,               // live source
            "do-timestamp", FALSE,              // don't create timestamps
            "block",        FALSE,              // do not block if queue is full
            "min-percent",  0,                  // always empty queue until 0%
            "max-bytes",    100000,  // max queue size
            NULL);
    gst_app_src_set_latency(GST_APP_SRC(mGstAppSrc), 0, 100000); // TODO performance: configuration? what is appropriate?

    decoder = gst_element_factory_make("vpudec", "decoder");
    if (!decoder)
    {
        mLog.info("Failed to initialise decoder");
        return VNCCommonDecoderErrorResourceError;
    }

    g_object_set(G_OBJECT(decoder), "low-latency", TRUE, NULL);
    g_object_set(G_OBJECT(decoder), "frame-plus", 2, NULL);
    g_object_set(G_OBJECT(decoder), "framedrop", FALSE, NULL);
    g_object_set(G_OBJECT(decoder), "framerate-nu", 30, NULL);
    g_object_set(G_OBJECT(decoder), "dis-reorder", TRUE, NULL);

    GstCaps* caps_vpu_dec = gst_caps_new_simple("video/x-h264",
                                            NULL);
    if (caps_vpu_dec != NULL)
    {
        g_object_set(G_OBJECT(decoder), "caps", caps_vpu_dec, NULL);
    }

    // gst_apx_sink display-width=800 display-height=480 sync=false qos=false max-lateness=3000000000

    mLog.info("GStreamerStream::setupGStreamer layer-id = %d surface-id = %d", mContext->container.layer_id, mContext->container.surface_id);

    sink = gst_element_factory_make("gst_apx_sink", "sink");
    g_object_set(G_OBJECT(sink), "display-width", mContext->container.screen_width, NULL);
    g_object_set(G_OBJECT(sink), "display-height", mContext->container.screen_height, NULL);
    g_object_set(G_OBJECT(sink), "x-offset", mContext->container.screen_x_offset, NULL);
    g_object_set(G_OBJECT(sink), "y-offset", mContext->container.screen_y_offset, NULL);
    g_object_set(G_OBJECT(sink), "sync", FALSE, NULL);
    g_object_set(G_OBJECT(sink), "qos", FALSE, NULL);
    g_object_set(G_OBJECT(sink), "max-lateness", 3000000000U, NULL);
    g_object_set(G_OBJECT(sink), "layer-id", mContext->container.layer_id, NULL);
    g_object_set(G_OBJECT(sink), "surface-id", mContext->container.surface_id, NULL);

//    sink = gst_element_factory_make("filesink", "sink");
//    g_object_set(G_OBJECT(sink), "location", "bar_snk.log", NULL);

    if (!sink)
    {
        mLog.info("Failed to initialise sink");
        return VNCCommonDecoderErrorResourceError;
    }
    mGstSink = sink;

    gst_bin_add_many(GST_BIN(mGstPipeline), mGstAppSrc, decoder, sink, NULL);

    gboolean linking = false;
    linking = gst_element_link(mGstAppSrc, decoder);
    mLog.info("Linking mGstAppSrc and decoder returns: %d", linking);
    linking = gst_element_link(decoder, sink);
    mLog.info("Linking decoder and sink: %d", linking);

    g_object_get(G_OBJECT (sink), "wl_display", &display, NULL);
    if (!display)
    {
        mLog.error("GStreamer not giving the display!!!");
    }
    else
    {
        mContext->container.is_display_ready = 1;
        mContext->container.p_wayland = display;
        mLog.info("GStreamer returned wayland display %p and assigned to %p", display, mContext->container.p_wayland);
    }

    return VNCCommonDecoderErrorNone;
}

void GStreamerStream::start()
{
    // Run pipeline
    GstStateChangeReturn result = gst_element_set_state(mGstPipeline, GST_STATE_PAUSED);
    if (result == GST_STATE_CHANGE_ASYNC)
    {
        result = gst_element_get_state(mGstPipeline, NULL, NULL, GST_SECOND);
    }
    gst_element_set_state(mGstPipeline, GST_STATE_PLAYING);

    mLog.info("GStreamer pipeline running");
}

void GStreamerStream::stop()
{
    // Pause pipeline
    gst_element_set_state(mGstPipeline, GST_STATE_PAUSED);
    mContext->container.is_display_ready = 0;
    mContext->container.is_streaming_stopped = 1;
    mLog.info("GStreamer pipeline paused");
}

void GStreamerStream::setVolume(vnc_uint32_t volume)
{
    if (mGstVolume)
    {
        volume = std::min(volume, static_cast<vnc_uint32_t>(100));
        mVolume = volume;
        gdouble volumeDouble = (gdouble)volume / (gdouble)100;
        mLog.info("Setting volume to %f", (float)volumeDouble);
        g_object_set(G_OBJECT(mGstVolume), "volume", volumeDouble, NULL);
    }
}

void GStreamerStream::setMuted(bool muted)
{
    if (mGstVolume)
    {
        mLog.info("Setting volume to %s", muted ? "TRUE" : "FALSE");
        g_object_set(G_OBJECT(mGstVolume), "mute", (gboolean)muted, NULL);
    }
}
